;==========================================================================
;Program Name:          GPCD9_Demo
;Applied body:          GPCD9 Series Body
;Programmer:            Arthur Chen
;Current Version:       V1.05
;Release Date:          2011/03/31
;History:
;       V1.04           Merge the libraries of SW_ChA and SW_ChB to a new library, SW_Channel_V100.lib
;                       Update the software channel library as "SW_Channel_V100.lib"
;                       Check if any single note/drum is played before turning off DAC
;                       Debug "F_CheckMIDIChVol", "F_StopSound" and dynamic allocation of channel in MIDI.asm
;                       Revise SW_Channel.asm to fit new library of software channel
;       V1.05           Update the software channel library as "SW_Channel_V102.lib"
;                       Revise dynamic allocation for playing a MIDI in MIDI.asm
;                       Debug "F_ChA_GetEvtAddr" and "F_ChB_GetEvtAddr" to fix calculation error of event address
;                       Add IO event function for SW_ChA, SW_ChB and MIDI modules
;                       Add new files "Module\IO_Event.asm" and "Inc\IO_Event.inc" for IO event
;==========================================================================
                        .SYNTAX 6502
                        .LINKLIST
                        .SYMBOLS
;==========================================================================
        .INCLUDE        GPCD9.inc
;==========================================================================
OFF:                    .EQU    0
ON:                     .EQU    1

SystemClock:            .EQU    16000000        ;Declaration of System Clock (Must be consistent with real system clock)
CPU_Clock:              .EQU    SystemClock/2   ;CPU Clock = System Clock / 2

D_MainVol:              .EQU    7FH             ;The main volume of SPU
D_MixerCtrl:            .EQU    FFH             ;Choose which channel is mixed to SPU
D_GainCtrl:             .EQU    D_GainLvD       ;Declare the default Gain Control of push-pull DAC

SPEECH:                 .EQU    ON              ;Declare Speech function with SPU
MIDI:                   .EQU    OFF             ;Declare MIDI function
SW_ChA:                 .EQU    OFF             ;Declare SW_ChA (The library must be enabled in Menu\Project\Setting\Link)
SW_ChB:                 .EQU    OFF             ;Declare SW_ChB (The library must be enabled in Menu\Project\Setting\Link)
INPUT:                  .EQU    ON              ;Declare INPUT function for scanning input key
FullSwing:              .EQU    OFF             ;Declare Full Swing of each channel is enabled/disabled
Comair5:                 .EQU    ON              ;Declare Comair5 is enabled/disabled
;==========================================================================
        .PAGE0
R_Command:              .DS     1               ;Command temporary
;==========================================================================
        .INCLUDE        Inc\System.inc
;==========================================================================
;User's Program (840H)
;==========================================================================
        .CODE

        .IF SW_ChA = ON || SW_ChB = ON
        .INCLUDE        Module\SW_Channel.asm
            .IF (SW_ChA = ON && ChA_Evt = ON && ChA_IOEvt = ON)||(SW_ChB = ON && ChB_Evt = ON && ChB_IOEvt = ON)
        .INCLUDE        Module\IO_Event.asm
            .ENDIF
        .ENDIF

        .IF MIDI = ON
        .INCLUDE        Module\MIDI.asm
            .IFNDEF D_IO_EventNum
                .IF MIDI_IOEvt = ON
        .INCLUDE        Module\IO_Event.asm
                .ENDIF
            .ENDIF
        .ENDIF

        .IF SPEECH = ON
        .INCLUDE        Module\Speech.asm
        .ENDIF

        .IF INPUT = ON
        .INCLUDE        Module\Input.asm
        .ENDIF

        .IF Comair5 = ON
        .INCLUDE        Module\Comair5.asm
        .ENDIF

        .INCLUDE        Tab\Channel.tab
;==========================================================================
        .CODE
V_RESET:
        %InitialSystem                  ;Initialize system status
        %ClearRAM       00H,FFH         ;Clear SRAM from 00H to FFH
        %ClearRAM       200H,2FFH       ;Clear SRAM from 200H to 2FFH
        %SetUpMainVol   D_MainVol       ;Set the main volume for SPU
        %SetUpMixer     D_MixerCtrl     ;Set the mixer for SPU
        %SetGainCtrl    D_GainCtrl      ;Set the gain control of push-pull DAC

        .IF MIDI = ON
        %SetUpTimeBase  D_ADSRTimeBase, D_NoteOffTimeBase       ;Set the timebases of envelope and note off
        %MIDIPowerUpInitial             ;Power-up initial for MIDI function
        .ENDIF

        .IF SPEECH = ON
        %SpeechPowerUpInitial           ;Power-up initial for SPEECH function
        .ENDIF

        .IF SW_ChA = ON
        %ChAInitial                     ;Initial for software channel A
        .ENDIF

        .IF SW_ChB = ON
        %ChBInitial                     ;Initial for software channel B
        .ENDIF

        .IF INPUT = ON
        %InputPowerUpInitial            ;Power-up initial for Input function
        .ENDIF

        .IF Comair5 = ON
        JSR     F_CA5_Initial
        .ENDIF

        .IFDEF D_IO_EventNum
        JSR     F_SW_PWM_PowerUpInitial ;Power-up initial for software PWMIOs
        .ENDIF

        JMP     L_PowerOnPlay
;==========================================================================
V_RESET_WAKEUP:
        %InitialSystem                  ;Initialize system status
        %SetUpMainVol   D_MainVol       ;Set the main volume for SPU
        %SetUpMixer     D_MixerCtrl     ;Set the mixer for SPU
        %SetGainCtrl    D_GainCtrl      ;Set the gain control of push-pull DAC

        .IF MIDI = ON
        %SetUpTimeBase  D_ADSRTimeBase, D_NoteOffTimeBase       ;Set the timebases of envelope and note off
        %MIDIWakeUpInitial              ;Wake-up initial for MIDI function
        .ENDIF

        .IF SPEECH = ON
        %SpeechWakeUpInitial            ;Wake-up initial for SPEECH function
        .ENDIF

        .IF SW_ChA = ON
        %ChAInitial                     ;Initial for software channel A
        .ENDIF

        .IF SW_ChB = ON
        %ChBInitial                     ;Initial for software channel B
        .ENDIF

        .IF INPUT = ON
        %InputWakeUpInitial             ;Wake-up initial for Input function
        .ENDIF

        .IF Comair5 = ON
        JSR     F_CA5_Initial
        .ENDIF

        .IFDEF D_IO_EventNum
        JSR     F_SW_PWM_WakeUpInitial  ;Wake-up initial for software PWMIOs
        .ENDIF
;==========================================================================
L_PowerOnPlay:
        %InitialIO                      ;Initialize all I/Os as input pull-low
        LDA     #00H
        STA     R_Command
L_MonitorLoop:
        %ClearWDT                       ;Clear watchdog timer

        .IF INPUT = ON
        JSR     F_Main                  ;User-defined function according to the state of Input function
        JSR     F_ServiceInput          ;Service for Input function
        .ENDIF

        .IF MIDI = ON
        JSR     F_ServiceMIDI           ;Service for MIDI sequencer
        JSR     F_CheckMidiDuration     ;Service for checking duration of MIDI notes
        JSR     F_CheckADSR             ;Service for envelope of MIDI notes
        .ENDIF

        .IF SW_ChA = ON
            .IF ChA_Evt = ON
        JSR     F_ChA_ServiceEvt        ;Service for speech event in SW_ChA
            .ENDIF
        .ENDIF

        .IF SW_ChB = ON
            .IF ChB_Evt = ON
        JSR     F_ChB_ServiceEvt        ;Service for speech event in SW_ChB
            .ENDIF
        .ENDIF

        JMP     L_MonitorLoop
;==========================================================================
        .IF INPUT = ON
F_Main:
        JSR     F_GetInput              ;Get input event on designated I/O port
        BCS     L_Key0
        JMP     L_ExitMain
L_Key0:                                 ;Check if the Key0 is triggered
        CMP     #00H
        BNE     L_Key1

        .IF Comair5 = ON
        JSR     F_CA5_IsActive
        BCS     L_Key0_Exit?
        INC     R_Command
        LDA     R_Command
        CMP		#79
        BCC		L_Next?
        LDA		#0
        STA     R_Command
L_Next?:        
        TAX
        JSR     F_CA5_SendCommand
        .ENDIF

        .IF SPEECH = ON
        %SetSpeechMode PCM              ;Set data format (ADPCM/PCM)
        TXA                             ;A = Speech file index (Refer to ResourceView window in FortisIDE or Resource window in G+IDE)
        LDX     #03H                    ;X = Sampling rate index (Refers to T_SpeechFreqTable in Speech.asm)
        LDY     #00H                    ;Y = Channel number (GPCD6:0~3; GPCD9:0~7)
        JSR     F_PlayHWSpeech          ;Play speech with SPU channel
        .ENDIF
L_Key0_Exit?:

        JMP     L_ExitMain

L_Key1:                                 ;Check if the Key1 is triggered
        CMP     #01H
        BNE     L_Key2

        .IF Comair5 = ON
        JSR     F_CA5_IsActive
        BCS     L_Key1_Exit?
        DEC     R_Command
        LDA     R_Command
        BPL		L_Next?
        LDA		#79
        STA     R_Command
L_Next?:        
        TAX
        JSR     F_CA5_SendCommand
        .ENDIF

        .IF SPEECH = ON
        %SetSpeechMode PCM              ;Set data format (ADPCM/PCM)
        TXA                             ;A = Speech file index (Refer to ResourceView window in FortisIDE or Resource window in G+IDE)
        LDX     #03H                    ;X = Sampling rate index (Refers to T_SpeechFreqTable in Speech.asm)
        LDY     #00H                    ;Y = Channel number (GPCD6:0~3; GPCD9:0~7)
        JSR     F_PlayHWSpeech          ;Play speech with SPU channel
        .ENDIF
L_Key1_Exit?:

        JMP     L_ExitMain

L_Key2:                                 ;Check if the Key2 is triggered
        CMP     #02H
        BNE     L_Key3

        JMP     L_ExitMain

L_Key3:                                 ;Check if the Key3 is triggered
        CMP     #03H
        BNE     L_Key4

        JMP     L_ExitMain

L_Key4:                                 ;Check if the Key4 is triggered
        CMP     #04H
        BNE     L_Key5

        JMP     L_ExitMain

L_Key5:                                 ;Check if the Key5 is triggered
        CMP     #05H
        BNE     L_Key6

        JMP     L_ExitMain

L_Key6:                                 ;Check if the Key6 is triggered
        CMP     #06H
        BNE     L_Key7

        JMP     L_ExitMain

L_Key7:                                 ;Check if the Key7 is triggered
        CMP     #07H
        BNE     L_ExitMain

L_ExitMain:
        RTS
        .ENDIF
;==========================================================================
;IRQ & NMI Interrupt Service Routine
;==========================================================================
V_IRQ_TMA:                              ;Timer A IRQ
        STA     P_TMA_Clr
        %PushAll

        .IF SW_ChA = ON
        JSR     F_ChA_Service           ;Service for software channel A on IRQ
        .ENDIF

        %PopAll
        RTI
;==========================================================================
V_IRQ_TMB:                              ;Timer B IRQ
        STA     P_TMB_Clr
        %PushAll

        .IF SW_ChB = ON
        JSR     F_ChB_Service           ;Service for software channel B on IRQ
        .ENDIF

        %PopAll
        RTI
;==========================================================================
V_IRQ_TMC:                              ;Timer C IRQ
        STA     P_TMC_Clr
        %PushAll

        .IFDEF D_IO_EventNum
        JSR     F_SW_PWM_Service
        .ENDIF

        .IF Comair5 = ON
        JSR     F_CA5_Service
        .ENDIF

        %PopAll
        RTI
;==========================================================================
V_IRQ_1024:                             ;Fcpu/1024 IRQ
        STA     P_1024_Clr
        %PushAll

        .IF MIDI = ON
        %MidiDataCounter                ;Timebase calculation for MIDI function
        .ENDIF

        %PopAll
        RTI
;==========================================================================
V_IRQ_4096:                             ;Fcpu/1024 IRQ
        STA     P_4096_Clr
        %PushAll

        .IF INPUT = ON
        %IntDecInputDebounceTimer       ;Debouncing for Input function
        .ENDIF

        %PopAll
        RTI
;==========================================================================
V_IRQ_16Hz:                             ;RTC 16Hz IRQ
        STA     P_16Hz_Clr
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_2Hz:                              ;RTC 2Hz IRQ
        STA     P_2Hz_Clr
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_KEY:                              ;Key Change IRQ
        STA     P_KEY_Clr
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_EXT:                              ;External Interrupt IRQ
        STA     P_EXT_Clr
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_SPU:                              ;SPU IRQ
        %PushAll

        .IF MIDI = ON
        JSR    F_SpuIrqMIDIService      ;Service for MIDI function on SPU IRQ
        .ENDIF

        .IF SPEECH = ON
        JSR    F_SpuIrqSpeechService    ;Service for SPEECH function on SPU IRQ
        .ENDIF

        %PopAll 
        RTI
;==========================================================================
V_IRQ_SPI:                              ;SPI IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_ADC:                              ;ADC IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_QD1F:                             ;QD1 Forward IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_QD1B:                             ;QD1 Backward IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_QD2F:                             ;QD2 Forward IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_QD2B:                             ;QD2 Backward IRQ
        %PushAll

        %PopAll
        RTI
;==========================================================================
V_IRQ_None:
        RTI
;==========================================================================
V_NMI:
        .IF Comair5 = ON
        PHA
        LDA     P_DAC2L                 ;Generate square wave on software channel 2
        EOR     #FFH
        STA     P_DAC2L
        LDA     P_DAC2H
        EOR     #FFH
        STA     P_DAC2H
        STA     P_TMB_Clr
        PLA
        .ENDIF
        RTI
;==========================================================================
        .IFDEF  GP_6502_ASM
        .INCLUDE        Res_tbl.inc    ;Index table of song and speech for G+IDE 6502 with assembly
        .ELSE
        .INCLUDE        Res_tbl.asm    ;Index table of song and speech for FortisIDE
        .ENDIF
;==========================================================================
L_Version:                              ;Module version
        .DB     "GPCD9_Demo_V1.05 2011/03/31"
;==========================================================================
;Succeeded label of Startup.asm
;==========================================================================
        .PUBLIC V_NMI           ;81AH, 81BH
        .PUBLIC V_RESET         ;81CH, 81DH
        .PUBLIC V_IRQ_None      ;81EH, 81FH

        .PUBLIC V_IRQ_TMA       ;820H, 821H
        .PUBLIC V_IRQ_TMB       ;822H, 823H
        .PUBLIC V_IRQ_TMC       ;824H, 825H
        .PUBLIC V_IRQ_1024      ;826H, 827H
        .PUBLIC V_IRQ_4096      ;828H, 829H
        .PUBLIC V_IRQ_16Hz      ;82AH, 82BH
        .PUBLIC V_IRQ_2Hz       ;82CH, 82DH
        .PUBLIC V_IRQ_KEY       ;82EH, 82FH
        .PUBLIC V_IRQ_EXT       ;830H, 831H
        .PUBLIC V_IRQ_SPU       ;832H, 833H
        .PUBLIC V_IRQ_SPI       ;834H, 835H
        .PUBLIC V_IRQ_ADC       ;836H, 837H
        .PUBLIC V_IRQ_QD1F      ;838H, 839H
        .PUBLIC V_IRQ_QD1B      ;83AH, 83BH
        .PUBLIC V_IRQ_QD2F      ;83CH, 83DH
        .PUBLIC V_IRQ_QD2B      ;83EH, 83FH

        .EXTERN T_StepTable
;==========================================================================
        .END
